function fr = frcasymp(b,a,tau,lgw1,lgw2)
% FRCASYMP Computes the Bode diagram asymptotes
%
%	fr = frcasymp(b,a,tau,lgw1,lgw2)
%
%	The frequency responses corresponding to the amplitude asymptotes of
%       G(s) = b(s)/a(s)*exp(-s*tau) between 10^lgw1 and 10^lgw2 are 
%	calculated. No attempt is made to calculate a phase asymptote, i.e. 
%	the phase of fr will be 0.
%
%       The output fr takes the form [ w  G(iw)].

% Kjell Gustafsson 
% LastEditDate : Fri Mar  9 01:36:29 1990
% Copyright (c) 1990 by Kjell Gustafsson and Department of Automatic Control,
% Lund Institute of Technology, Lund, SWEDEN

w1 = 10^lgw1;
w2 = 10^lgw2;
  
% Find out how many poles and zeros at the origin

inda = find(a>eps);
na = length(a)-inda(length(inda));
indb = find(b>eps);
nb = length(b)-indb(length(indb));

% Calculate the slope of the low frequency asymptote and 
% its magnitude for w = 1

slope0 = nb-na;
mag0 = b(indb(length(indb)))/a(inda(length(inda)));

if na==length(a)-1 & nb==length(b)-1
  % only poles and zeros at the origin
  fr = [w1 mag0*w1^slope0; w2 mag0*w2^slope0];
else
  % Calculate non origin poles and zeros

  p = roots(a(1:length(a)-na));
  z = roots(b(1:length(b)-nb));

  % Sort poles and zeros in magnitude order, calculate asymptote slopes
 
  [wvec,ind] = sort(abs([p; z]));
  slope = [-ones(length(p),1); ones(length(z),1)];
  slope = cumsum(slope(ind)) + slope0;

  % Calculate magnitudes for each frequency in wvec
  
  mag = zeros(size(wvec));
  mag(1) = mag0*wvec(1)^slope0;
  for k = 2:length(wvec)
    mag(k) = mag(k-1)*(wvec(k)/wvec(k-1))^slope(k-1);
  end
  
  % Strip out the region between lgw1 and lgw2
  
  if w1<=wvec(1)
    mag = [mag(1)*(w1/wvec(1))^slope0; mag];
    wvec = [w1; wvec];
  else
    ind = find(wvec>w1);
    mag = [mag(ind(1))*(w1/wvec(ind(1)))^slope(ind(1)-1); mag(ind)];
    wvec = [w1; wvec(ind)];
  end
  
  ind = find(wvec<w2);
  mag = [mag(ind); mag(ind(length(ind)))*...
	   (w2/wvec(ind(length(ind))))^slope(ind(length(ind))-1)];
  wvec = [wvec(ind); w2];
  
  fr = [wvec mag];
end
