function locus = rootloc(b,a,k1,k2,dsmax,ucirc);
% ROOTLOC Calculates rootlocus
%
%	locus = rootloc(a,b,k1,k2,dsmax,ucirc)
%	locus = rootloc(a,b,k1,k2,dsmax)
%
%	Calculates the root locus of a(s) + k*b(s) for k1 < k < k2. The
% 	increment in k is calculated such that abs(ds) < dsmax for the 
%       fastest root branch. This is done using the implicit function 
%	theorem. 
%
%       The roots and the corresponding k-values are stored in locus, with
%       the gains in the first column. 
%
% 	All k (k1 < k < k2) giving rise to multiple poles are calculated
%       and and the corresponding roots are included in locus.
%
%	If ucirc (optional) is supplied (any value) only the roots inside
%	the unit circle will be forced to obey dsmax.

% Kjell Gustafsson, derived from routines by Mats Lilja
% LastEditDate : Thu May 30 09:15:25 1991
% Copyright (c) 1990 by Kjell Gustafsson and Department of Automatic Control,
% Lund Institute of Technology, Lund, SWEDEN

na = length(a);
nb = length(b);
d = na - nb;
if d>=0,
  b = [0*(1:d) b];
else
  a = [0*(1:-d) a];
end
da = derpoly(a);
db = derpoly(b);
k = k1;
epsi = 1e-10;
i = 1;
while k<=k2,
  kvec(i) = k;
  ri = roots(a+k*b)';
  rvec(i,:) = ri;
  if nargin==6
    ind = find(abs(ri)<=1);
    ri = ri(ind);
  end
  fs = polyval(da+k*db,ri);
  fk = polyval(b,ri);
  dk = dsmax*min(abs(fs./fk))+epsi;
  k =  k + dk;
  i =  i + 1;
end;
locus = [kvec' rvec];

% multiple roots

dab = conv(da,b);
adb = conv(a,db);
if nb>1,
  mpol = dab - adb;
else
  mpol = dab;
end;
multroots = roots(mpol);
bmult = polyval(b,multroots);
amult = polyval(a,multroots);
ind = find(bmult~=0);
kmult = -amult(ind)./bmult(ind);
mr = [];
for j=1:length(kmult),
  if abs(imag(kmult(j)))<1e-9,
    kj = real(kmult(j));
    if kj>k1 & kj<k2,
      index = sum(locus(:,1)<kj);
      locus = [locus(1:index,:); [kj roots(a+kj*b)'];...
        locus(index+1:length(locus(:,1)),:)];
    end;
  end;
end;
